{
This file Copyright 2000 (c) CDF, Inc.
Written By: Edward Flick (Directrix1@yahoo.com)
Use at your own risk!
}

unit ExpressionEval;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, db;

const
  whitespace=[' ','+'];

type
  RegCustFunction = function(ins: array of String): String;
  ObjCustFunction = function(ins: array of String): String of Object;
  TCustFunction = class
    public
      { Public declarations }
      internalName: String;
      reg: boolean;
      regFunc: RegCustFunction;
      objFunc: ObjCustFunction;
    end;

{
type
  CompiledChunkType = (cctLiteral, cctFieldVal, cctRegFunc, cctObjFunc);
  TCompiledChunk = record
}

type
  TExpressionEval = class(TComponent)
  protected
    { Protected declarations }
    FSource: TDataset;
    function eval(expr: String;var idx: integer): String;
    function processQuote(expr: String;var idx:integer;quotechar: char): String;
    function processChar(expr: String;var idx: integer): String;
    function processOther(expr: String;var idx: integer): String;
    procedure skipWhiteSpace(expr: String;var idx:integer);
    //Predefined Custom functions
    function CUpper(ins: array of String): String;
    function CSubstr(ins: array of String): String;
    function CLen(ins: array of String): String;
    function CLeft(ins: array of String): String;
    function CRight(ins: array of String): String;
    function CPadL(ins: array of String): String;
    function CPadR(ins: array of String): String;
    function CAllTrim(ins: array of String): String;
    function CAt(ins: array of String): String;
  public
    { Public declarations }
    FCustFunctions: array of TCustFunction;
    function evaluate(expr: String): String;
    procedure registerFunction(FuncName: String; theFunc: RegCustFunction); overload; virtual;
    procedure registerFunction(FuncName: String; theFunc: ObjCustFunction); reintroduce; overload;
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property Source: TDataset read FSource write FSource;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Data Access', [TExpressionEval]);
end;

function TExpressionEval.processQuote(expr: String;var idx:integer;quotechar: char): String;
var
  done: boolean;
begin
try
  result:='';
  inc(idx);
  done:=false;
  repeat
  if expr[idx]=quotechar then //A quotechar has been encountered
    begin                    //in the literal
    inc(idx);
    if expr[idx]=quotechar then
      begin
      result:=result+quotechar; //At double quotechars denotes an embedded
      inc(idx);    //quotechar in the literal
      end
    else
      done:=true; //At Single quotechar denotes end of literal
    end
  else
    begin
    result:=result+expr[idx]; //Just another char in the literal
    inc(idx);
    end;
  until (done);
except
  result:='';
  raise;
  end;
end;

function TExpressionEval.processChar(expr: String;var idx: integer): String;
var
  done: boolean;
begin
try
  result:='';
  inc(idx);
  done:=false;
  while (not done) do
    begin
    if idx>length(expr) then
      done:=true
    else
      if (ord(expr[idx])>=ord('0')) and (ord(expr[idx])<=ord('9')) then
        begin
        result:=result+expr[idx];
        inc(idx);
        end
      else
        done:=true;
    end;
  result:=char(strtoint(result));
except
  result:='';
  raise;
  end;
end;

function TExpressionEval.processOther(expr: String;var idx: integer): String;
var
  done: boolean;
  j,k: integer;
  params: array of String;
  key: string;
begin
try
  result:='';
  key:='';
  done:=false;
  setLength(params,0);
  while (not done) do
    begin
    if (idx>length(expr)) then
      done:=true
    else
      if (not (expr[idx] in whitespace)) and
         (expr[idx]<>')') and
         (expr[idx]<>',') and
         (expr[idx]<>'(') then
        begin
        key:=key+expr[idx];
        inc(idx);
        end
      else
        done:=true;
    end;
  skipWhiteSpace(expr,idx);
  if idx<=length(expr) then
    begin
    if expr[idx]='(' then
      begin
      k:=-1;
      for j:=0 to length(FCustFunctions)-1 do   //Check custom methods registry
        if (FCustFunctions[j].internalName=UpperCase(key)) then
          k:=j;
      if k=-1 then
        raise Exception(''''+key+''' is not a registered function.');
      inc(idx);
      repeat
        skipWhiteSpace(expr,idx);
        if expr[idx]=',' then
          begin
          inc(idx);
          result:=eval(expr,idx);
          setLength(params,length(params)+1);
          params[length(params)-1]:=result;
          end
        else
          begin
          result:=eval(expr,idx);
          setLength(params,length(params)+1);
          params[length(params)-1]:=result;
          end;
      until expr[idx]=')';
      inc(idx);
      if FCustFunctions[k].reg then
        result:=FCustFunctions[k].regFunc(params)
      else
        result:=FCustFunctions[k].objFunc(params);
      end
    else
      begin
      if FSource.FindField(key)<>nil then
        result:=FSource.FieldByName(key).asString   //Try fieldval method
      else
        raise Exception.create(''''+key+''' is not a valid field or function.');
      end;
    end
  else
    if FSource.FindField(key)<>nil then
      result:=FSource.FieldByName(key).asString   //Try fieldval method
    else
      raise Exception.create(''''+key+''' is not a valid field or function.');
except
  result:='';
  raise;
  end;
end;

procedure TExpressionEval.skipWhiteSpace(expr: String;var idx:integer);
var
  done: boolean;
begin
done:=false;
while (not done) do        //Skip placeholders
  begin
  if idx>length(expr) then
    done:=true
  else
    if expr[idx] in whitespace then
      inc(idx)
    else
      done:=true;
  end;
end;

//Evaluate expr using the Source dataset, as a fieldval source
function TExpressionEval.eval(expr: String;var idx: integer): String;
var
  res, tempin: String;
begin
res:='';
tempin:=expr;
skipWhiteSpace(tempin,idx);
while ((idx<=length(tempin)) and (not (tempin[idx] in [',',')']))) do
  begin
  if tempin[idx]='''' then //An apostrophe has been encountered
    begin                    //in the string
    res:=res+processQuote(tempin,idx,'''');
    end
  else
    begin
    if tempin[idx]='#' then //Ordinal Value code
      begin
      res:=res+processChar(tempin,idx);
      end
    else
      begin                  //Get Other
      res:=res+processOther(tempin,idx);
      end;
    end;
  skipWhiteSpace(tempin,idx);
  end;
result:=res; //Return it!
end;

function TExpressionEval.evaluate(expr: String): String;
var
  index: integer;
begin
index:=1;
try
  result:=eval(expr,index);
except
  on e: Exception do
    begin
    result:='';
    raise Exception.create('Error while evaluating expression: '+chr(13)+chr(10)+
    expr+chr(13)+chr(10)+
    '@char('+inttostr(index)+'), With Error:'+chr(13)+chr(10)+
    e.message);         //Very handy exception routine, I believe
    end;
  end;
end;

procedure TExpressionEval.registerFunction(FuncName: String; theFunc: RegCustFunction);
var
  temp: TCustFunction;
  j: integer;
begin
temp:=nil;
if length(FCustFunctions)>0 then
  for j:=0 to length(FCustFunctions) -1 do
    if UpperCase(FuncName)=FCustFunctions[j].internalName then
      temp:=FCustFunctions[j];
if temp=nil then
  begin
  temp:=TCustFunction.Create();
  setLength(FCustFunctions, length(FCustFunctions)+1);
  FCustFunctions[length(FCustFunctions)-1]:=temp;
  end;
temp.internalName:=UpperCase(FuncName);
temp.regFunc:=theFunc;
temp.reg:=true;
end;

procedure TExpressionEval.registerFunction(FuncName: String; theFunc: ObjCustFunction);
var
  temp: TCustFunction;
  j: integer;
begin
temp:=nil;
if length(FCustFunctions)>0 then
  for j:=0 to length(FCustFunctions) -1 do
    if UpperCase(FuncName)=FCustFunctions[j].internalName then
      temp:=FCustFunctions[j];
if temp=nil then
  begin
  temp:=TCustFunction.Create();
  setLength(FCustFunctions, length(FCustFunctions)+1);
  FCustFunctions[length(FCustFunctions)-1]:=temp;
  end;
temp.internalName:=UpperCase(FuncName);
temp.objFunc:=theFunc;
temp.reg:=false;
end;

constructor TExpressionEval.Create(Owner: TComponent);
begin
inherited;
setLength(FCustFunctions,0);
registerFunction('UPPER',CUpper);
registerFunction('SUBSTR',CSubstr);
registerFunction('LEN',CLen);
registerFunction('LEFT',CLeft);
registerFunction('RIGHT',CRight);
registerFunction('PADL',CPadL);
registerFunction('PADR',CPadR);
registerFunction('ALLTRIM',CAllTrim);
registerFunction('AT',CAt);
end;

destructor TExpressionEval.Destroy;
var
  j: integer;
begin
if length(FCustFunctions)>0 then
  for j:=0 to length(FCustFunctions)-1 do
    FCustFunctions[j].Free;
inherited;
end;

// Default functions
function TExpressionEval.CUpper(ins: array of String): String;
begin
  result:=UpperCase(ins[0]);
end;

function TExpressionEval.CSubstr(ins: array of String): String;
begin
  if length(ins)<3 then
    result:=copy(ins[0],strtoint(ins[1]),2000000000)
  else
    result:=copy(ins[0],strtoint(ins[1]),strtoint(ins[2]));
end;

function TExpressionEval.CLen(ins: array of String): String;
begin
  result:=inttostr(length(ins[0]));
end;

function TExpressionEval.CLeft(ins: array of String): String;
begin
  result:=copy(ins[0],1,strtoint(ins[1]));
end;

function TExpressionEval.CRight(ins: array of String): String;
begin
  result:=copy(ins[0],length(ins[0])-strtoint(ins[1])+1,strtoint(ins[1]));
end;

function TExpressionEval.CPadL(ins: array of String): String;
var
  dc: char;
begin
  if length(ins)<3 then
    dc:=' '
  else
    dc:=ins[3][1];
  if length(ins[0])>=strtoint(ins[1]) then
    result:=ins[0]
  else
    result:=StringOfChar(dc,strtoint(ins[1])-length(ins[0]))+ins[0];
end;

function TExpressionEval.CPadR(ins: array of String): String;
var
  dc: char;
begin
  if length(ins)<3 then
    dc:=' '
  else
    dc:=ins[3][1];
  if length(ins[0])>=strtoint(ins[1]) then
    result:=ins[0]
  else
    result:=ins[0]+StringOfChar(dc,strtoint(ins[1])-length(ins[0]));
end;

function TExpressionEval.CAllTrim(ins: array of String): String;
begin
  result:=trim(ins[0]);
end;

function TExpressionEval.CAt(ins: array of String): String;
begin
  result:=inttostr(Pos(ins[0],ins[1]));
end;

end.
